/**
* \file: blacklist.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include "control/blacklist.h"

#include <stdlib.h>
#include <string.h>
#include <fnmatch.h>

#include "utils/logger.h"
#include "utils/helper.h"
#include "utils/string_set.h"
#include "model/partition.h"
#include "control/automounter.h"

static const char *blacklisted_system_virtual_devices[]=
{
		"devtmpfs",
		"devpts",
		"proc",
		"sysfs",
		"securityfs",
		"tmpfs",
		"cgroup",
		"systemd-1",
		NULL
};

static string_set_t *supported_fs_list=NULL;

static string_set_t *partition_blacklist=NULL;

static string_set_t *device_blacklist=NULL;

error_code_t blacklist_init(void)
{
	error_code_t result;

	result=string_set_create(&partition_blacklist,1024,1);
	if (result==RESULT_OK)
		result=string_set_create(&device_blacklist,1024,1);
	if (result==RESULT_OK)
		result=string_set_create(&supported_fs_list,100,1);

	if (result!=RESULT_OK)
		blacklist_deinit();

	return result;
}

void blacklist_deinit(void)
{
	if (partition_blacklist!=NULL)
	{
		string_set_free(partition_blacklist);
		partition_blacklist=NULL;
	}

	if (device_blacklist!=NULL)
	{
		string_set_free(device_blacklist);
		device_blacklist=NULL;
	}

	if (supported_fs_list!=NULL)
	{
		string_set_free(supported_fs_list);
		supported_fs_list=NULL;
	}
}

bool blacklist_is_device_blacklisted(const char* device_id)
{
	if (device_id!=NULL)
		return string_set_does_item_match(device_blacklist, device_id);

	return false;
}

bool blacklist_is_partition_blacklisted(const char* partition_id)
{
	if (partition_id!=NULL)
		return string_set_does_item_match(partition_blacklist, partition_id);

	return false;
}

bool blacklist_is_fs_supported(const char *fstype)
{
	return string_set_does_item_match(supported_fs_list,fstype);
}

void blacklist_add_blacklisted_device(const char* device_id)
{
	if (string_set_add(&device_blacklist,device_id)!=RESULT_OK)
		automounter_die_on_resource_issues();
	logger_log_debug("BLACKLIST - Add device to black list: %s",device_id);
}

void blacklist_add_blacklisted_partition(const char* partition_id)
{
	if (string_set_add(&partition_blacklist,partition_id)!=RESULT_OK)
		automounter_die_on_resource_issues();
	logger_log_debug("BLACKLIST - Add partition to black list: %s",partition_id);
}

void blacklist_add_supported_fs_array(char* fstype_array)
{
	char *token;
	token=strtok(fstype_array," \t");

	while(token != NULL)
	{
		if (string_set_add(&supported_fs_list,token)!=RESULT_OK)
			automounter_die_on_resource_issues();
		logger_log_debug("BLACKLIST - Mark file system as supported: %s",token);
		token=strtok(NULL," \t");
	}
}

bool blacklist_is_system_device(const char *mount_src)
{
	int itr;

	for (itr=0;blacklisted_system_virtual_devices[itr]!=NULL;itr++)
		if (strcmp(blacklisted_system_virtual_devices[itr],mount_src)==0)
			return true;

	return false;
}
